<?php

function load_links() {
    $file = LINKS_FILE;
    if (!file_exists($file)) {
        file_put_contents($file, '{}');
    }
    $json = file_get_contents($file);
    return json_decode($json, true);
}

function save_links($links) {
    $file = LINKS_FILE;
    file_put_contents($file, json_encode($links, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
}

function load_users() {
    $file = USERS_FILE;
    if (!file_exists($file)) {
        file_put_contents($file, '{}');
    }
    $json = file_get_contents($file);
    return json_decode($json, true);
}

function save_users($users) {
    $file = USERS_FILE;
    file_put_contents($file, json_encode($users, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
}

function load_states() {
    $file = STATES_FILE;
    if (!file_exists($file)) {
        file_put_contents($file, '{}');
    }
    $json = file_get_contents($file);
    return json_decode($json, true);
}

function save_states($states) {
    $file = STATES_FILE;
    file_put_contents($file, json_encode($states, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
}

function set_user_state($user_id, $state) {
    $states = load_states();
    $states[$user_id] = $state;
    save_states($states);
}

function get_user_state($user_id) {
    $states = load_states();
    return isset($states[$user_id]) ? $states[$user_id] : null;
}

function add_user($user_id, $username) {
    $users = load_users();
    if (!isset($users[$user_id])) {
        $users[$user_id] = [
            'username' => $username,
            'joined_at' => date('Y-m-d H:i:s')
        ];
        save_users($users);
    }
}

function generate_unique_id($user_id) {
    $encoded_id = base64_encode($user_id);
    return $encoded_id . '_' . bin2hex(random_bytes(4));
}

function load_channels() {
    $file = __DIR__ . '/channels.json';
    if (!file_exists($file)) {
        file_put_contents($file, json_encode(['required_channels' => []]));
    }
    $json = file_get_contents($file);
    return json_decode($json, true);
}

function save_channels($channels) {
    $file = __DIR__ . '/channels.json';
    file_put_contents($file, json_encode($channels, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
}

function add_required_channel($channel_data) {
    $channels = load_channels();
    
    // اطمینان از وجود ساختار جدید
    if (!isset($channels['required_channels'])) {
        $channels['required_channels'] = [];
    }
    
    // تبدیل ساختار قدیمی به جدید (یکبار)
    if (!empty($channels['required_channels']) && is_string($channels['required_channels'][0])) {
        $old_channels = $channels['required_channels'];
        $channels['required_channels'] = [];
        foreach ($old_channels as $old_channel) {
            $channels['required_channels'][] = [
                'id' => $old_channel,
                'name' => '@' . $old_channel,
                'link' => 'https://t.me/' . $old_channel
            ];
        }
    }
    
    // پارس کردن اطلاعات ورودی
    if (is_string($channel_data)) {
        // فرمت قدیمی - فقط username
        $channel_id = trim(str_replace('@', '', $channel_data));
        $channel_info = [
            'id' => $channel_id,
            'name' => '@' . $channel_id,
            'link' => 'https://t.me/' . $channel_id
        ];
    } else {
        $channel_info = $channel_data;
    }
    
    // چک کردن تکراری نبودن
    foreach ($channels['required_channels'] as $existing) {
        if ($existing['id'] === $channel_info['id']) {
    return false;
}
    }
    
    $channels['required_channels'][] = $channel_info;
    save_channels($channels);
    return true;
}

function remove_required_channel($channel_id) {
    $channels = load_channels();
    
    // نرمال‌سازی ورودی: حذف @ و فضای خالی
    $channel_id = trim(str_replace('@', '', $channel_id));
    
    // پشتیبانی از ساختار قدیمی
    if (!empty($channels['required_channels']) && is_string($channels['required_channels'][0])) {
        $key = array_search($channel_id, $channels['required_channels']);
    if ($key !== false) {
        unset($channels['required_channels'][$key]);
        $channels['required_channels'] = array_values($channels['required_channels']);
        save_channels($channels);
        return true;
        }
        return false;
    }
    
    // ساختار جدید - بررسی ID با حالت‌های مختلف
    foreach ($channels['required_channels'] as $key => $channel) {
        $stored_id = $channel['id'];
        
        // مقایسه مستقیم (با تبدیل نوع)
        if ($stored_id == $channel_id) {
            unset($channels['required_channels'][$key]);
            $channels['required_channels'] = array_values($channels['required_channels']);
            save_channels($channels);
            return true;
        }
        
        // اگر ورودی عددی است، امتحان کن با علامت منفی
        if (is_numeric($channel_id)) {
            // اگر کاربر علامت منفی رو فراموش کرده
            if ($stored_id == "-$channel_id") {
                unset($channels['required_channels'][$key]);
                $channels['required_channels'] = array_values($channels['required_channels']);
                save_channels($channels);
                return true;
            }
            // یا اگر کاربر علامت منفی زده ولی ID در دیتابیس بدون منفی هست
            if (strpos($channel_id, '-') === 0 && $stored_id == substr($channel_id, 1)) {
                unset($channels['required_channels'][$key]);
                $channels['required_channels'] = array_values($channels['required_channels']);
                save_channels($channels);
                return true;
            }
        }
    }
    return false;
}

function check_user_membership($user_id) {
    $channels = load_channels();
    if (empty($channels['required_channels'])) {
        return true;
    }

    $not_joined = [];
    foreach ($channels['required_channels'] as $channel) {
        // پشتیبانی از ساختار قدیمی و جدید
        if (is_string($channel)) {
            $channel_id = '@' . $channel;
            $channel_name = $channel;
        } else {
            $channel_id = $channel['id'];
            // اگر شناسه عددی است، @ نمی‌گذاریم
            if (!is_numeric($channel_id) && strpos($channel_id, '-') !== 0) {
                $channel_id = '@' . $channel_id;
            }
            $channel_name = $channel['name'] ?? $channel_id;
        }
        
        $url = "https://api.telegram.org/bot" . BOT_TOKEN . "/getChatMember";
        $data = [
            'chat_id' => $channel_id,
            'user_id' => $user_id
        ];
        
        try {
            $context = stream_context_create([
                'http' => [
                    'timeout' => 5,
                    'ignore_errors' => true
                ]
            ]);
            
            $response = file_get_contents($url . '?' . http_build_query($data), false, $context);
            if ($response === false) {
                error_log("خطا در بررسی عضویت کاربر $user_id در کانال $channel_name: خطای اتصال");
                continue;
            }
            
            $result = json_decode($response, true);
            if (!$result || !isset($result['ok'])) {
                error_log("خطا در بررسی عضویت کاربر $user_id در کانال $channel_name: پاسخ نامعتبر");
                continue;
            }
            
            if (!$result['ok']) {
                error_log("خطا در بررسی عضویت کاربر $user_id در کانال $channel_name: " . ($result['description'] ?? 'خطای ناشناخته'));
                continue;
            }
            
            if (!in_array($result['result']['status'], ['member', 'administrator', 'creator'])) {
                $not_joined[] = is_string($channel) ? $channel : $channel['id'];
            }
        } catch (Exception $e) {
            error_log("خطا در بررسی عضویت کاربر $user_id در کانال $channel_name: " . $e->getMessage());
            continue;
        }
    }
    
    return empty($not_joined);
}

function add_link($unique_id, $original_link, $user_id) {
    $links = load_links();
    $links[$unique_id] = [
        'original_link' => $original_link,
        'user_id' => $user_id,
        'created_at' => date('Y-m-d H:i:s')
    ];
    save_links($links);
}

function get_link($unique_id) {
    $links = load_links();
    return isset($links[$unique_id]) ? $links[$unique_id]['original_link'] : null;
}

function send_message($chat_id, $text, $reply_markup = null, $disable_web_page_preview = true) {
    $url = "https://api.telegram.org/bot" . BOT_TOKEN . "/sendMessage";
    $data = [
        'chat_id' => $chat_id,
        'text' => $text,
        'parse_mode' => 'HTML',
        'disable_web_page_preview' => $disable_web_page_preview
    ];
    if ($reply_markup) {
        $data['reply_markup'] = json_encode($reply_markup);
    }
    $response = file_get_contents($url . '?' . http_build_query($data));
    $result = json_decode($response, true);
    return $result['ok'] ? $result['result'] : null;
}

function is_admin($user_id) {
    return in_array($user_id, ADMIN_IDS);
}

function get_stats() {
    $links = load_links();
    $users = load_users();
    $user_count = count($users);
    $link_count = count($links);
    return [
        'user_count' => $user_count,
        'link_count' => $link_count
    ];
}

function broadcast_message($message, $chat_id = null, $message_id = null) {
    $users = load_users();
    $batch_size = 100; // تعداد ارسال در هر بچ
    $total_users = count($users);
    $processed_file = __DIR__ . '/broadcast_processed.json';
    
    // بارگذاری لیست کاربران پردازش شده
    if (file_exists($processed_file)) {
        $processed_data = json_decode(file_get_contents($processed_file), true) ?: [];
    } else {
        $processed_data = [];
    }

    // اطمینان از وجود همه کلیدهای مورد نیاز
    $processed_data = array_merge([
        'message_id' => null,
        'processed_users' => [],
        'total' => 0,
        'success' => 0,
        'failed' => 0,
        'status_chat_id' => null,
        'status_message_id' => null
    ], $processed_data);

    // اگر این یک پیام جدید است یا اطلاعات ناقص است
    if ($processed_data['message_id'] !== $message || 
        $processed_data['status_chat_id'] !== $chat_id || 
        $processed_data['status_message_id'] !== $message_id) {
        
        $processed_data = [
            'message_id' => $message,
            'processed_users' => [],
            'total' => $total_users,
            'success' => 0,
            'failed' => 0,
            'status_chat_id' => $chat_id,
            'status_message_id' => $message_id
        ];
    }

    $current_batch = 0;
    foreach ($users as $user_id => $user) {
        // اگر کاربر قبلاً پیام را دریافت کرده، رد کن
        if (in_array($user_id, $processed_data['processed_users'])) {
            continue;
        }

        // محدودیت تعداد در هر بچ
        if ($current_batch >= $batch_size) {
            break;
        }

        $url = "https://api.telegram.org/bot" . BOT_TOKEN . "/sendMessage";
        $data = [
            'chat_id' => $user_id,
            'text' => $message,
            'parse_mode' => 'HTML'
        ];
        
        $response = @file_get_contents($url . '?' . http_build_query($data));
        $result = json_decode($response, true);
        
        if ($result && $result['ok']) {
            $processed_data['success']++;
        } else {
            $processed_data['failed']++;
            // اگر کاربر ربات را بلاک کرده یا حذف کرده
            if (isset($result['error_code']) && in_array($result['error_code'], [403, 400])) {
                unset($users[$user_id]);
                save_users($users);
            }
        }

        // اضافه کردن کاربر به لیست پردازش شده
        $processed_data['processed_users'][] = $user_id;
        $current_batch++;
    }

    // ذخیره وضعیت پردازش
    file_put_contents($processed_file, json_encode($processed_data, JSON_PRETTY_PRINT));

    // اگر پنل مدیریت باز است، آمار را برگردان
    if ($processed_data['status_chat_id'] && $processed_data['status_message_id']) {
        $remaining = $total_users - count($processed_data['processed_users']);
        $progress_text = "📊 وضعیت ارسال همگانی:\n\n";
        $progress_text .= "👥 تعداد افراد ارسالی: " . count($processed_data['processed_users']) . "\n";
        $progress_text .= "⏱ زمان تخمینی باقیمانده: " . ceil($remaining / $batch_size) . " دقیقه\n\n";
        $progress_text .= "✅ موفق: " . $processed_data['success'] . "\n";
        $progress_text .= "❌ ناموفق: " . $processed_data['failed'] . "\n";
        
        $reply_markup = null;
        if ($remaining > 0) {
            $reply_markup = [
                'inline_keyboard' => [
                    [['text' => '🔄 بروزرسانی آمار', 'callback_data' => 'refresh_broadcast_stats']]
                ]
            ];
        }
        
        edit_message($processed_data['status_chat_id'], $processed_data['status_message_id'], $progress_text, $reply_markup);
    }

    // اگر همه کاربران پردازش شده‌اند، فایل را پاک کن
    if (count($processed_data['processed_users']) >= $total_users) {
        @unlink($processed_file);
        return true;
    }

    return false;
}

function forward_message_to_all($from_chat_id, $message_id, $chat_id = null, $status_message_id = null) {
    $users = load_users();
    $batch_size = 100; // تعداد ارسال در هر بچ
    $total_users = count($users);
    $processed_file = __DIR__ . '/forward_processed.json';
    
    // بارگذاری لیست کاربران پردازش شده
    if (file_exists($processed_file)) {
        $processed_data = json_decode(file_get_contents($processed_file), true) ?: [];
    } else {
        $processed_data = [];
    }

    // اطمینان از وجود همه کلیدهای مورد نیاز
    $processed_data = array_merge([
        'message_id' => null,
        'processed_users' => [],
        'total' => 0,
        'success' => 0,
        'failed' => 0,
        'status_chat_id' => null,
        'status_message_id' => null
    ], $processed_data);

    $message_key = $from_chat_id . ':' . $message_id;

    // اگر این یک پیام جدید است یا اطلاعات ناقص است
    if ($processed_data['message_id'] !== $message_key || 
        $processed_data['status_chat_id'] !== $chat_id || 
        $processed_data['status_message_id'] !== $status_message_id) {
        
        $processed_data = [
            'message_id' => $message_key,
            'processed_users' => [],
            'total' => $total_users,
            'success' => 0,
            'failed' => 0,
            'status_chat_id' => $chat_id,
            'status_message_id' => $status_message_id
        ];
    }

    $current_batch = 0;
    foreach ($users as $user_id => $user) {
        // اگر کاربر قبلاً پیام را دریافت کرده، رد کن
        if (in_array($user_id, $processed_data['processed_users'])) {
            continue;
        }

        // محدودیت تعداد در هر بچ
        if ($current_batch >= $batch_size) {
            break;
        }

        $url = "https://api.telegram.org/bot" . BOT_TOKEN . "/forwardMessage";
        $data = [
            'chat_id' => $user_id,
            'from_chat_id' => $from_chat_id,
            'message_id' => $message_id
        ];
        
        $response = @file_get_contents($url . '?' . http_build_query($data));
        $result = json_decode($response, true);
        
        if ($result && $result['ok']) {
            $processed_data['success']++;
        } else {
            $processed_data['failed']++;
            // اگر کاربر ربات را بلاک کرده یا حذف کرده
            if (isset($result['error_code']) && in_array($result['error_code'], [403, 400])) {
                unset($users[$user_id]);
                save_users($users);
            }
        }

        // اضافه کردن کاربر به لیست پردازش شده
        $processed_data['processed_users'][] = $user_id;
        $current_batch++;
    }

    // ذخیره وضعیت پردازش
    file_put_contents($processed_file, json_encode($processed_data, JSON_PRETTY_PRINT));

    // اگر پنل مدیریت باز است، آمار را برگردان
    if ($processed_data['status_chat_id'] && $processed_data['status_message_id']) {
        $remaining = $total_users - count($processed_data['processed_users']);
        $progress_text = "📊 وضعیت فوروارد همگانی:\n\n";
        $progress_text .= "👥 تعداد افراد ارسالی: " . count($processed_data['processed_users']) . "\n";
        $progress_text .= "⏱ زمان تخمینی باقیمانده: " . ceil($remaining / $batch_size) . " دقیقه\n\n";
        $progress_text .= "✅ موفق: " . $processed_data['success'] . "\n";
        $progress_text .= "❌ ناموفق: " . $processed_data['failed'] . "\n";
        
        $reply_markup = null;
        if ($remaining > 0) {
            $reply_markup = [
                'inline_keyboard' => [
                    [['text' => '🔄 بروزرسانی آمار', 'callback_data' => 'refresh_forward_stats']]
                ]
            ];
        }
        
        edit_message($processed_data['status_chat_id'], $processed_data['status_message_id'], $progress_text, $reply_markup);
    }

    // اگر همه کاربران پردازش شده‌اند، فایل را پاک کن
    if (count($processed_data['processed_users']) >= $total_users) {
        @unlink($processed_file);
        return true;
    }

    return false;
}

function delete_message($chat_id, $message_id) {
    $url = "https://api.telegram.org/bot" . BOT_TOKEN . "/deleteMessage";
    $data = [
        'chat_id' => $chat_id,
        'message_id' => $message_id
    ];
    
    // Use POST method instead of GET for more reliability
    $options = [
        'http' => [
            'method' => 'POST',
            'header' => 'Content-Type: application/x-www-form-urlencoded',
            'content' => http_build_query($data),
            'ignore_errors' => true
        ]
    ];
    
    $context = stream_context_create($options);
    $response = @file_get_contents($url, false, $context);
    
    if ($response === false) {
        error_log(sprintf(
            "خطا در حذف پیام %d در چت %s: خطای اتصال",
            $message_id,
            $chat_id
        ));
        return false;
    }
    
    $result = json_decode($response, true);
    
    // اگر پیام قبلاً حذف شده یا پیدا نشد، موفقیت آمیز در نظر بگیریم
    if (!$result['ok'] && isset($result['error_code']) && $result['error_code'] === 400) {
        error_log(sprintf(
            "نتیجه حذف پیام %d در چت %s: %s",
            $message_id,
            $chat_id,
            $result['description'] ?? 'خطای ناشناخته'
        ));
        return true;
    }
    
    return $result['ok'] ?? false;
}

function send_temporary_message($chat_id, $text, $reply_markup = null, $delete_after = null) {
    // اگر زمان حذف مشخص نشده، از تنظیمات استفاده کن
    if ($delete_after === null) {
        $delete_after = get_auto_delete_time();
    }
    
    $message = send_message($chat_id, $text, $reply_markup);
    if ($message) {
        // Store deletion info
        $delete_at = time() + $delete_after;
        register_deletion($chat_id, $message['message_id'], $delete_at);
        
        // Schedule immediate background deletion
        schedule_deletion($chat_id, $message['message_id'], $delete_after);
    }
    return $message;
}

function schedule_deletion($chat_id, $message_id, $delay) {
    // Create a background process to handle deletion
    $cmd = sprintf(
        'curl --max-time 10 -s -X POST "https://api.telegram.org/bot%s/deleteMessage" -d "chat_id=%s&message_id=%d" > /dev/null 2>&1',
        BOT_TOKEN,
        $chat_id,
        $message_id
    );
    
    // Use different methods based on OS
    if (PHP_OS === 'WINNT') {
        // Windows
        pclose(popen(sprintf('start /B php -r "sleep(%d); %s;"', $delay, $cmd), 'r'));
    } else {
        // Linux/Unix
        shell_exec(sprintf('(sleep %d; %s) > /dev/null 2>&1 &', $delay, $cmd));
        
        // Also schedule through at command as backup
        $at_time = date('H:i', time() + $delay);
        $at_cmd = sprintf('echo "%s" | at %s > /dev/null 2>&1', $cmd, $at_time);
        shell_exec($at_cmd);
    }
}

function register_deletion($chat_id, $message_id, $delete_at) {
    $file = __DIR__ . '/message_deletions.json';
    if (!file_exists($file)) {
        file_put_contents($file, json_encode(['deletions' => []]));
    }
    
    $data = json_decode(file_get_contents($file), true) ?: ['deletions' => []];
    
    // Remove any existing deletion for this message
    $data['deletions'] = array_filter($data['deletions'], function($item) use ($chat_id, $message_id) {
        return !($item['chat_id'] == $chat_id && $item['message_id'] == $message_id);
    });
    
    // Add new deletion
    $data['deletions'][] = [
        'chat_id' => $chat_id,
        'message_id' => $message_id,
        'delete_at' => $delete_at,
        'created_at' => time()
    ];
    
    file_put_contents($file, json_encode($data, JSON_PRETTY_PRINT));
}

function cleanup_messages() {
    $file = __DIR__ . '/message_deletions.json';
    if (!file_exists($file)) {
        return;
    }
    
    $data = json_decode(file_get_contents($file), true) ?: ['deletions' => []];
    $now = time();
    $remaining = [];
    $modified = false;
    
    foreach ($data['deletions'] as $deletion) {
        if ($deletion['delete_at'] <= $now) {
            // Try to delete the message
            $result = delete_message($deletion['chat_id'], $deletion['message_id']);
            if ($result) {
                $modified = true;
                continue;
            }
            
            // اگر حذف ناموفق بود و پیام بیش از 5 دقیقه از موعد حذفش گذشته
            if ($now - $deletion['delete_at'] > 300) {
                $modified = true;
                continue;
            }
        }
        $remaining[] = $deletion;
    }
    
    if ($modified) {
        $data['deletions'] = $remaining;
        file_put_contents($file, json_encode($data, JSON_PRETTY_PRINT));
    }
}

function edit_message($chat_id, $message_id, $text, $reply_markup = null) {
    static $last_message = [];
    $message_key = $chat_id . '_' . $message_id;
    
    // ساخت محتوای پیام برای مقایسه
    $current_content = [
        'text' => $text,
        'reply_markup' => $reply_markup ? json_encode($reply_markup) : null
    ];
    
    // اگر پیام تغییر نکرده، آپدیت نکن
    if (isset($last_message[$message_key]) && 
        $last_message[$message_key]['text'] === $current_content['text'] && 
        $last_message[$message_key]['reply_markup'] === $current_content['reply_markup']) {
        return true;
    }
    
    $url = "https://api.telegram.org/bot" . BOT_TOKEN . "/editMessageText";
    $data = [
        'chat_id' => $chat_id,
        'message_id' => $message_id,
        'text' => $text,
        'parse_mode' => 'HTML'
    ];
    if ($reply_markup) {
        $data['reply_markup'] = json_encode($reply_markup);
    }

    $options = [
        'http' => [
            'method' => 'POST',
            'header' => 'Content-Type: application/x-www-form-urlencoded',
            'content' => http_build_query($data),
            'ignore_errors' => true
        ]
    ];
    
    $context = stream_context_create($options);
    $response = @file_get_contents($url, false, $context);
    
    if ($response === false) {
        error_log("خطا در ویرایش پیام: خطای اتصال");
        return false;
    }
    
    $result = json_decode($response, true);
    
    if (!$result['ok']) {
        // اگر پیام تغییر نکرده، خطا نیست
        if (isset($result['description']) && strpos($result['description'], 'message is not modified') !== false) {
            $last_message[$message_key] = $current_content;
            return true;
        }
        error_log("خطا در ویرایش پیام: " . ($result['description'] ?? 'خطای ناشناخته'));
        return false;
    }
    
    // ذخیره آخرین محتوای پیام
    $last_message[$message_key] = $current_content;
    return $result;
}

function answer_callback_query($callback_query_id, $text = null, $show_alert = false) {
    $url = "https://api.telegram.org/bot" . BOT_TOKEN . "/answerCallbackQuery";
    $data = [
        'callback_query_id' => $callback_query_id,
        'show_alert' => $show_alert
    ];
    if ($text) {
        $data['text'] = $text;
    }
    $response = file_get_contents($url . '?' . http_build_query($data));
    return json_decode($response, true);
}

function get_links_paginated($page = 1, $per_page = 5) {
    $links = load_links();
    $total = count($links);
    $total_pages = ceil($total / $per_page);
    $page = max(1, min($page, $total_pages));
    
    $start = ($page - 1) * $per_page;
    $paginated = array_slice($links, $start, $per_page, true);
    
    return [
        'links' => $paginated,
        'total' => $total,
        'per_page' => $per_page,
        'current_page' => $page,
        'total_pages' => $total_pages
    ];
}

function get_detailed_stats() {
    $stats = get_stats();
    $users = load_users();
    $links = load_links();
    
    // آمار کاربران فعال در 24 ساعت گذشته
    $active_users = 0;
    $now = time();
    foreach ($users as $user) {
        if (isset($user['last_activity']) && ($now - strtotime($user['last_activity'])) <= 86400) {
            $active_users++;
        }
    }
    
    // آمار لینک‌های ایجاد شده در 24 ساعت گذشته
    $recent_links = 0;
    foreach ($links as $link) {
        if ($now - strtotime($link['created_at']) <= 86400) {
            $recent_links++;
        }
    }
    
    return [
        'total_users' => $stats['user_count'],
        'total_links' => $stats['link_count'],
        'active_users_24h' => $active_users,
        'new_links_24h' => $recent_links
    ];
}

function get_channel_keyboard() {
    $channels = load_channels();
    $keyboard = [];
    
    foreach ($channels['required_channels'] as $channel) {
        // پشتیبانی از ساختار قدیمی و جدید
        if (is_string($channel)) {
            $keyboard[] = [
                ['text' => "@$channel", 'url' => "https://t.me/$channel"]
            ];
        } else {
            $text = isset($channel['name']) ? $channel['name'] : "کانال";
            $url = $channel['link'] ?? "https://t.me/" . $channel['id'];
            $keyboard[] = [
                ['text' => $text, 'url' => $url]
            ];
        }
    }
    
    $keyboard[] = [
        ['text' => "✅ بررسی عضویت", 'callback_data' => 'check_membership']
    ];
    
    return [
        'inline_keyboard' => $keyboard
    ];
}

function get_start_message() {
    $file = __DIR__ . '/start_message.json';
    if (!file_exists($file)) {
        $default_data = [
            'type' => 'text',
            'message' => "👋 سلام! لینک خود را ارسال کنید تا محافظت شود.",
            'photo' => null,
            'inline_buttons' => [],
            'forward_from_chat_id' => null,
            'forward_message_id' => null
        ];
        file_put_contents($file, json_encode($default_data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
        return $default_data;
    }
    $data = json_decode(file_get_contents($file), true);
    
    // پشتیبانی از فرمت قدیمی
    if (isset($data['message']) && is_string($data['message']) && !isset($data['type'])) {
        $old_message = $data['message'];
        $data = [
            'type' => 'text',
            'message' => $old_message,
            'photo' => null,
            'inline_buttons' => [],
            'forward_from_chat_id' => null,
            'forward_message_id' => null
        ];
        file_put_contents($file, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    }
    
    return $data;
}

function save_start_message($message_data) {
    $file = __DIR__ . '/start_message.json';
    
    // اگر رشته ساده است، به ساختار جدید تبدیل کن
    if (is_string($message_data)) {
        $message_data = [
            'type' => 'text',
            'message' => $message_data,
            'photo' => null,
            'inline_buttons' => [],
            'forward_from_chat_id' => null,
            'forward_message_id' => null
        ];
    }
    
    return file_put_contents($file, json_encode($message_data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
}

function send_start_message($chat_id, $custom_keyboard = null) {
    $start_data = get_start_message();
    
    // اگر نوع پیام متن یا عکس است
    $text = $start_data['message'] ?? "👋 سلام! لینک خود را ارسال کنید تا محافظت شود.";
    $photo = $start_data['photo'] ?? null;
    $inline_buttons = $start_data['inline_buttons'] ?? [];
    
    // ساخت کیبورد شیشه‌ای
    $reply_markup = null;
    if (!empty($inline_buttons)) {
        $keyboard_rows = [];
        foreach ($inline_buttons as $btn) {
            $keyboard_rows[] = [
                ['text' => $btn['text'], 'url' => $btn['url']]
            ];
        }
        $reply_markup = ['inline_keyboard' => $keyboard_rows];
    }
    
    // اگر کیبورد سفارشی داریم و inline button نداریم
    if (!$reply_markup && $custom_keyboard) {
        $reply_markup = $custom_keyboard;
    }
    
    // اگر عکس داریم
    if ($photo) {
        $url = "https://api.telegram.org/bot" . BOT_TOKEN . "/sendPhoto";
        $data = [
            'chat_id' => $chat_id,
            'photo' => $photo,
            'caption' => $text,
            'parse_mode' => 'HTML'
        ];
        
        if ($reply_markup) {
            $data['reply_markup'] = json_encode($reply_markup);
        }
        
        $response = file_get_contents($url . '?' . http_build_query($data));
        $result = json_decode($response, true);
        return $result['ok'] ? $result['result'] : null;
    }
    
    // فقط متن
    return send_message($chat_id, $text, $reply_markup);
}

function get_forced_join_message() {
    $file = __DIR__ . '/forced_join_message.json';
    if (!file_exists($file)) {
        $default_message = "⚠️ لطفاً ابتدا در کانال‌های زیر عضو شوید:\n\n";
        file_put_contents($file, json_encode(['message' => $default_message], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
        return $default_message;
    }
    $data = json_decode(file_get_contents($file), true);
    return $data['message'] ?? "⚠️ لطفاً ابتدا در کانال‌های زیر عضو شوید:\n\n";
}

function save_forced_join_message($message) {
    $file = __DIR__ . '/forced_join_message.json';
    return file_put_contents($file, json_encode(['message' => $message], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
}

function get_auto_delete_time() {
    $file = __DIR__ . '/auto_delete_settings.json';
    if (!file_exists($file)) {
        $default_time = 60; // 60 ثانیه (1 دقیقه)
        file_put_contents($file, json_encode(['delete_after' => $default_time], JSON_PRETTY_PRINT));
        return $default_time;
    }
    $data = json_decode(file_get_contents($file), true);
    return $data['delete_after'] ?? 60;
}

function save_auto_delete_time($seconds) {
    // محدود کردن بین 10 ثانیه تا 300 ثانیه (5 دقیقه)
    $seconds = max(10, min(300, intval($seconds)));
    $file = __DIR__ . '/auto_delete_settings.json';
    return file_put_contents($file, json_encode(['delete_after' => $seconds], JSON_PRETTY_PRINT));
}

// مدیریت دکمه‌های سفارشی
function load_custom_buttons() {
    $file = __DIR__ . '/custom_buttons.json';
    if (!file_exists($file)) {
        file_put_contents($file, json_encode(['buttons' => []], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    }
    $data = json_decode(file_get_contents($file), true);
    return $data['buttons'] ?? [];
}

function save_custom_buttons($buttons) {
    $file = __DIR__ . '/custom_buttons.json';
    return file_put_contents($file, json_encode(['buttons' => $buttons], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
}

function add_custom_button($button_data) {
    $buttons = load_custom_buttons();
    
    // ایجاد ID یکتا برای دکمه
    $button_data['id'] = uniqid('btn_');
    $button_data['created_at'] = date('Y-m-d H:i:s');
    
    $buttons[] = $button_data;
    save_custom_buttons($buttons);
    return $button_data['id'];
}

function get_custom_button($button_id) {
    $buttons = load_custom_buttons();
    foreach ($buttons as $button) {
        if ($button['id'] === $button_id) {
            return $button;
        }
    }
    return null;
}

function get_custom_button_by_name($button_name) {
    $buttons = load_custom_buttons();
    foreach ($buttons as $button) {
        if ($button['name'] === $button_name) {
            return $button;
        }
    }
    return null;
}

function delete_custom_button($button_id) {
    $buttons = load_custom_buttons();
    $new_buttons = array_filter($buttons, function($btn) use ($button_id) {
        return $btn['id'] !== $button_id;
    });
    save_custom_buttons(array_values($new_buttons));
    return count($buttons) !== count($new_buttons);
}

function get_custom_buttons_keyboard() {
    $buttons = load_custom_buttons();
    if (empty($buttons)) {
        return null;
    }
    
    $keyboard = [];
    $row = [];
    
    foreach ($buttons as $index => $button) {
        $row[] = ['text' => $button['name']];
        
        // هر 2 دکمه یک ردیف
        if (count($row) == 2 || $index == count($buttons) - 1) {
            $keyboard[] = $row;
            $row = [];
        }
    }
    
    return [
        'keyboard' => $keyboard,
        'resize_keyboard' => true,
        'one_time_keyboard' => false
    ];
}